document.addEventListener('DOMContentLoaded', function() {
    const audio = document.getElementById('audioPlayer');
    const playPauseBtn = document.getElementById('playPauseBtn');
    const prevBtn = document.getElementById('prevBtn');
    const nextBtn = document.getElementById('nextBtn');
    const volumeControl = document.getElementById('volumeControl');
    const visualizer = document.getElementById('visualizer');
    const petalsContainer = document.getElementById('petals-container');
    const progressBar = document.getElementById('progressBar');
    const currentTimeDisplay = document.getElementById('currentTime');
    const durationDisplay = document.getElementById('duration');
    const trackList = document.getElementById('trackList');
    const togglePlaylistBtn = document.getElementById('togglePlaylist');
    const closePlaylistBtn = document.getElementById('closePlaylist');
    const playlistSidebar = document.getElementById('playlistSidebar');
    
    let tracks = [];
    let currentTrackIndex = 0;
    let audioContext;
    let analyser;
    let trackCategories = {};

    // Создаем волны для визуализатора
    const waveCount = 8;
    for (let i = 0; i < waveCount; i++) {
        const wave = document.createElement('div');
        wave.className = 'wave';
        wave.style.width = `${100 + i * 30}px`;
        wave.style.height = `${100 + i * 30}px`;
        visualizer.appendChild(wave);
    }
    const waves = document.querySelectorAll('.wave');
    
    // Управление видимостью плейлиста
    togglePlaylistBtn.addEventListener('click', function() {
        playlistSidebar.classList.add('visible');
    });
    
    closePlaylistBtn.addEventListener('click', function() {
        playlistSidebar.classList.remove('visible');
    });
    
    document.addEventListener('click', function(e) {
        if (!playlistSidebar.contains(e.target) && 
            e.target !== togglePlaylistBtn && 
            !togglePlaylistBtn.contains(e.target)) {
            playlistSidebar.classList.remove('visible');
        }
    });
    
    // Загрузка списка треков
    async function loadTrackList() {
        try {
            const response = await fetch('tracklist.json');
            tracks = await response.json();
            
            trackCategories = tracks.reduce((acc, track) => {
                const folder = track.path.split('/')[1];
                if (!acc[folder]) acc[folder] = [];
                acc[folder].push(track);
                return acc;
            }, {});
            
            renderTrackList();
            
            if (tracks.length > 0) {
                loadTrack(currentTrackIndex);
            }
        } catch (error) {
            console.error('Error loading track list:', error);
            tracks = [
                { path: 'm/ambient/sleep.ogg', title: 'Sleep', artist: 'Ambient' },
                { path: 'm/nature/rain.ogg', title: 'Rain', artist: 'Nature' }
            ];
            renderTrackList();
            loadTrack(currentTrackIndex);
        }
    }
    
    // Отображение списка треков с сворачиваемыми категориями
    function renderTrackList() {
        trackList.innerHTML = '';
        
        for (const [category, categoryTracks] of Object.entries(trackCategories)) {
            const categoryElement = document.createElement('div');
            categoryElement.className = 'track-category';
            
            const categoryName = document.createElement('div');
            categoryName.className = 'category-name';
            categoryName.textContent = category;
            
            const tracksElement = document.createElement('div');
            tracksElement.className = 'category-tracks';
            
            categoryTracks.forEach((track, index) => {
                const globalIndex = tracks.findIndex(t => t.path === track.path);
                const trackElement = document.createElement('div');
                trackElement.className = 'track-item';
                if (globalIndex === currentTrackIndex) trackElement.classList.add('active');
                trackElement.innerHTML = `
                    <span class="track-title">${track.title}</span>
                    <span class="track-artist">${track.artist || ''}</span>
                `;
                trackElement.addEventListener('click', () => {
                    currentTrackIndex = globalIndex;
                    loadTrack(currentTrackIndex);
                    playlistSidebar.classList.remove('visible');
                });
                tracksElement.appendChild(trackElement);
            });
            
            categoryName.addEventListener('click', () => {
                categoryName.classList.toggle('expanded');
                tracksElement.classList.toggle('visible');
            });
            
            categoryElement.appendChild(categoryName);
            categoryElement.appendChild(tracksElement);
            trackList.appendChild(categoryElement);
        }
    }
    
    // Загрузка трека по индексу
    function loadTrack(index) {
        if (index < 0 || index >= tracks.length) return;
        
        currentTrackIndex = index;
        const track = tracks[index];
        audio.src = track.path;
        audio.load();
        
        renderTrackList();
        
        if (!audioContext) {
            initAudioContext();
        }
        
        audio.play().then(() => {
            playPauseBtn.textContent = '❚❚';
            playPauseBtn.style.background = 'linear-gradient(135deg, #ff6b8b, #ff8c00)';
        }).catch(e => console.log('Autoplay prevented:', e));
    }
    
    // Инициализация аудио контекста
    function initAudioContext() {
        audioContext = new (window.AudioContext || window.webkitAudioContext)();
        analyser = audioContext.createAnalyser();
        analyser.fftSize = 256;
        
        const source = audioContext.createMediaElementSource(audio);
        source.connect(analyser);
        analyser.connect(audioContext.destination);
    }
    
    // Переключение треков
    function playNextTrack() {
        const nextIndex = (currentTrackIndex + 1) % tracks.length;
        loadTrack(nextIndex);
    }
    
    function playPrevTrack() {
        const prevIndex = (currentTrackIndex - 1 + tracks.length) % tracks.length;
        loadTrack(prevIndex);
    }
    
    // Обработчики кнопок
    prevBtn.addEventListener('click', playPrevTrack);
    nextBtn.addEventListener('click', playNextTrack);
    audio.addEventListener('ended', playNextTrack);
    
    // Управление плеером
    playPauseBtn.addEventListener('click', function() {
        if (audioContext && audioContext.state === 'suspended') {
            audioContext.resume();
        }
        
        if (audio.paused) {
            audio.play()
                .then(() => {
                    playPauseBtn.textContent = '❚❚';
                    playPauseBtn.style.background = 'linear-gradient(135deg, #ff6b8b, #ff8c00)';
                })
                .catch(e => console.error('Playback failed:', e));
        } else {
            audio.pause();
            playPauseBtn.textContent = '▶';
            playPauseBtn.style.background = 'linear-gradient(135deg, #ff8c00, #ff6b8b)';
        }
    });
    
    volumeControl.addEventListener('input', function() {
        audio.volume = this.value;
    });
    
    // Прогресс бар и время
    function formatTime(seconds) {
        const mins = Math.floor(seconds / 60);
        const secs = Math.floor(seconds % 60);
        return `${mins}:${secs < 10 ? '0' : ''}${secs}`;
    }
    
    function updateProgress() {
        if (!isNaN(audio.duration)) {
            const progress = (audio.currentTime / audio.duration) * 100;
            progressBar.value = progress;
            currentTimeDisplay.textContent = formatTime(audio.currentTime);
        }
    }
    
    progressBar.addEventListener('input', function() {
        const seekTime = (this.value / 100) * audio.duration;
        audio.currentTime = seekTime;
    });
    
    audio.addEventListener('loadedmetadata', function() {
        durationDisplay.textContent = formatTime(audio.duration);
    });
    
    audio.addEventListener('timeupdate', updateProgress);
    
    // Лепестки сакуры
    const petals = [];
    const petalCount = 50;
    
    function createPetal(resetPosition = false) {
        const petal = document.createElement('div');
        petal.className = 'petal';
        
        const size = 5 + Math.random() * 10;
        petal.style.width = `${size}px`;
        petal.style.height = `${size}px`;
        
        if (resetPosition) {
            petal.x = window.innerWidth / 2;
            petal.y = window.innerHeight / 2;
        } else {
            petal.x = Math.random() * window.innerWidth;
            petal.y = Math.random() * window.innerHeight;
        }
        
        petal.velocity = {
            x: (Math.random() - 0.5) * 0.5,
            y: (Math.random() - 0.5) * 0.5,
            rot: (Math.random() - 0.5) * 2
        };
        
        petal.rotation = Math.random() * 360;
        petal.opacity = 0.5 + Math.random() * 0.5;
        
        petalsContainer.appendChild(petal);
        petals.push(petal);
        
        updatePetalPosition(petal);
    }
    
    for (let i = 0; i < petalCount; i++) {
        createPetal();
    }
    
    function updatePetalPosition(petal) {
        petal.style.left = `${petal.x}px`;
        petal.style.top = `${petal.y}px`;
        petal.style.transform = `rotate(${petal.rotation}deg)`;
        petal.style.opacity = petal.opacity;
    }
    
    function animatePetals(intensity) {
        petals.forEach(petal => {
            const speedMultiplier = 1 + intensity * 3;
            
            petal.x += petal.velocity.x * speedMultiplier;
            petal.y += petal.velocity.y * speedMultiplier;
            petal.rotation += petal.velocity.rot;
            
            if (petal.x < -50 || petal.x > window.innerWidth + 50 ||
                petal.y < -50 || petal.y > window.innerHeight + 50) {
                
                if (petal.x < -50) petal.x = window.innerWidth + 49;
                if (petal.x > window.innerWidth + 50) petal.x = -49;
                if (petal.y < -50) petal.y = window.innerHeight + 49;
                if (petal.y > window.innerHeight + 50) petal.y = -49;
            }
            
            petal.opacity = 0.5 + intensity * 0.5;
            
            updatePetalPosition(petal);
        });
    }
    
    function visualize() {
        requestAnimationFrame(visualize);
        
        if (!analyser) return;
        
        const bufferLength = analyser.frequencyBinCount;
        const dataArray = new Uint8Array(bufferLength);
        analyser.getByteFrequencyData(dataArray);
        
        let sum = 0;
        for (let i = 0; i < bufferLength; i++) {
            sum += dataArray[i];
        }
        const average = sum / bufferLength;
        const intensity = average / 255;
        
        animatePetals(intensity);
        
        waves.forEach((wave, index) => {
            const waveIndex = Math.floor(index * (bufferLength / waveCount));
            const waveValue = dataArray[waveIndex] / 255;
            
            const baseSize = 100 + index * 30;
            const pulseSize = baseSize * (1 + waveValue * 0.2);
            
            wave.style.width = `${pulseSize}px`;
            wave.style.height = `${pulseSize}px`;
            wave.style.borderWidth = `${1 + waveValue * 3}px`;
            wave.style.opacity = 0.3 + waveValue * 0.7;
        });
    }
    
    window.addEventListener('resize', function() {
        petals.forEach(petal => {
            if (petal.x > window.innerWidth) petal.x = window.innerWidth;
            if (petal.y > window.innerHeight) petal.y = window.innerHeight;
            updatePetalPosition(petal);
        });
    });
    
    // Запускаем приложение
    loadTrackList();
    visualize();
});