<?php
// TaskerPHP — современный многопользовательский трекер задач (single-file)
// Требования: PHP 8.0+ (рекомендуется 8.1+), SQLite3, включенные PDO и pdo_sqlite, сессии.
// Поместите файл в директорию и откройте в браузере. При первом запуске создаст БД в папке storage/.

declare(strict_types=1);

// ---------------------- CONFIG ---------------------- //
const APP_NAME   = 'TaskerPHP';
const CSRF_KEY   = 'csrf_token';
const SESSION_NAME = 'tasker_session';

// Храним БД в подпапке storage (обычно разрешена запись на shared-хостинге)
const DB_DIR  = __DIR__ . '/storage';
const DB_PATH = DB_DIR . '/tasker.sqlite';

session_name(SESSION_NAME);
session_start();

// Отладка (true только временно)
$DEBUG = false;
if ($DEBUG) { ini_set('display_errors', '1'); error_reporting(E_ALL); }

// Создаём папку для БД и проверяем окружение
(function () {
    // 1) Проверка расширения pdo_sqlite
    if (!extension_loaded('pdo_sqlite')) {
        http_response_code(500);
        echo "Отсутствует расширение pdo_sqlite. Включите его в настройках PHP (php.ini / панель хостинга).";
        exit;
    }

    // 2) Создаём папку storage при необходимости
    if (!is_dir(DB_DIR)) {
        if (!mkdir(DB_DIR, 0775, true) && !is_dir(DB_DIR)) {
            http_response_code(500);
            echo "Не удалось создать папку для БД: " . DB_DIR;
            exit;
        }
    }

    // 3) Проверяем, можно ли писать в неё
    if (!is_writable(DB_DIR)) {
        http_response_code(500);
        echo "Нет прав на запись в " . DB_DIR . ". Выставьте права 0755/0775 и владельца веб-сервера.";
        exit;
    }

    // 4) open_basedir может запрещать путь
    $ob = (string)ini_get('open_basedir');
    if ($ob) {
        $ok = false;
        $real = realpath(DB_DIR) ?: DB_DIR;
        foreach (explode(PATH_SEPARATOR, $ob) as $allowed) {
            $allowed = rtrim($allowed, '/');
            if ($allowed === '') continue;
            if (strpos($real, $allowed) === 0) { $ok = true; break; }
        }
        if (!$ok) {
            http_response_code(500);
            echo "open_basedir запрещает запись в " . DB_DIR . ". Разрешённые пути: {$ob}";
            exit;
        }
    }
})();

// ---------------------- UTILS ---------------------- //
function db(): PDO {
    static $pdo = null;
    if ($pdo === null) {
        $needMigrate = !file_exists(DB_PATH);
        // Инициализация без именованных аргументов — совместимо с PHP 8.0+
        $pdo = new PDO('sqlite:' . DB_PATH);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        $pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);
        $pdo->exec('PRAGMA foreign_keys = ON');
        if ($needMigrate) migrate($pdo);
    }
    return $pdo;
}

function migrate(PDO $pdo): void {
    $pdo->exec(<<<SQL
        CREATE TABLE users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            email TEXT UNIQUE NOT NULL,
            name TEXT NOT NULL,
            password_hash TEXT NOT NULL,
            is_admin INTEGER NOT NULL DEFAULT 0,
            created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
        );
    SQL);

    $pdo->exec(<<<SQL
        CREATE TABLE projects (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            description TEXT,
            owner_id INTEGER NOT NULL,
            created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY(owner_id) REFERENCES users(id) ON DELETE CASCADE
        );
    SQL);

    $pdo->exec(<<<SQL
        CREATE TABLE tasks (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            project_id INTEGER NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            status TEXT NOT NULL DEFAULT 'todo', -- todo|in_progress|done|blocked
            priority TEXT NOT NULL DEFAULT 'normal', -- low|normal|high|urgent
            due_date TEXT,
            reporter_id INTEGER NOT NULL,
            assignee_id INTEGER,
            labels TEXT,
            created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TEXT,
            FOREIGN KEY(project_id) REFERENCES projects(id) ON DELETE CASCADE,
            FOREIGN KEY(reporter_id) REFERENCES users(id) ON DELETE SET NULL,
            FOREIGN KEY(assignee_id) REFERENCES users(id) ON DELETE SET NULL
        );
    SQL);

    $pdo->exec(<<<SQL
        CREATE TABLE comments (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            task_id INTEGER NOT NULL,
            author_id INTEGER NOT NULL,
            body TEXT NOT NULL,
            created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY(task_id) REFERENCES tasks(id) ON DELETE CASCADE,
            FOREIGN KEY(author_id) REFERENCES users(id) ON DELETE SET NULL
        );
    SQL);

    $pdo->exec(<<<SQL
        CREATE TABLE events (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            entity TEXT NOT NULL, -- task|project|user
            entity_id INTEGER NOT NULL,
            action TEXT NOT NULL, -- create|update|comment|status|assign
            meta TEXT,
            created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
            actor_id INTEGER,
            FOREIGN KEY(actor_id) REFERENCES users(id) ON DELETE SET NULL
        );
    SQL);
}

function now(): string { return (new DateTimeImmutable('now'))->format('Y-m-d H:i:s'); }

function e(?string $s): string { return htmlspecialchars($s ?? '', ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }

function redirect(string $url): never { header('Location: ' . $url); exit; }

function current_user(): ?array { return $_SESSION['user'] ?? null; }

function require_login(): array {
    $u = current_user();
    if (!$u) redirect('?action=login');
    return $u;
}

function is_post(): bool { return ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST'; }

function csrf_token(): string {
    if (empty($_SESSION[CSRF_KEY])) {
        $_SESSION[CSRF_KEY] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_KEY];
}

function check_csrf(): void {
    $token = $_POST['csrf'] ?? '';
    if (!$token || !hash_equals($_SESSION[CSRF_KEY] ?? '', $token)) {
        http_response_code(400);
        echo 'CSRF token invalid';
        exit;
    }
}

function flash(?string $msg = null, string $type = 'info'): ?array {
    if ($msg !== null) { $_SESSION['flash'][] = [$type, $msg]; return null; }
    $f = $_SESSION['flash'] ?? [];
    unset($_SESSION['flash']);
    return $f;
}

function option_selected(string $value, string $current): string { return $value === $current ? 'selected' : ''; }

function log_event(PDO $db, string $entity, int $entity_id, string $action, array $meta = []): void {
    $stmt = $db->prepare('INSERT INTO events(entity, entity_id, action, meta, created_at, actor_id) VALUES(?,?,?,?,?,?)');
    $stmt->execute([$entity, $entity_id, $action, json_encode($meta, JSON_UNESCAPED_UNICODE), now(), current_user()['id'] ?? null]);
}

// ---------------------- AUTH ---------------------- //
function handle_register(): void {
    if (is_post()) {
        check_csrf();
        $email = trim((string)($_POST['email'] ?? ''));
        $name = trim((string)($_POST['name'] ?? ''));
        $pass = (string)($_POST['password'] ?? '');
        if (!$email || !$name || !$pass) {
            flash('Заполните все поля', 'danger');
        } else {
            try {
                $hash = password_hash($pass, PASSWORD_DEFAULT);
                $stmt = db()->prepare('INSERT INTO users(email,name,password_hash,is_admin,created_at) VALUES(?,?,?,?,?)');
                $stmt->execute([$email, $name, $hash, 0, now()]);
                flash('Успешная регистрация! Теперь войдите.', 'success');
                redirect('?action=login');
            } catch (PDOException $e) {
                flash('Ошибка: возможно, такой email уже зарегистрирован.', 'danger');
            }
        }
    }
    render('Регистрация', function () {
        ?>
        <form method="post" class="card">
            <?php csrf_field(); ?>
            <h2>Регистрация</h2>
            <label>Email<input type="email" name="email" required></label>
            <label>Имя<input type="text" name="name" required></label>
            <label>Пароль<input type="password" name="password" required></label>
            <button class="btn primary" type="submit">Создать аккаунт</button>
            <p class="muted">Уже есть аккаунт? <a href="?action=login">Войти</a></p>
        </form>
        <?php
    });
}

function handle_login(): void {
    if (is_post()) {
        check_csrf();
        $email = trim((string)($_POST['email'] ?? ''));
        $pass = (string)($_POST['password'] ?? '');
        $stmt = db()->prepare('SELECT * FROM users WHERE email = ?');
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        if ($user && password_verify($pass, $user['password_hash'])) {
            session_regenerate_id(true);
            $_SESSION['user'] = $user;
            flash('С возвращением, ' . e($user['name']) . '!', 'success');
            redirect('index.php');
        } else {
            flash('Неверный email или пароль', 'danger');
        }
    }
    render('Вход', function () {
        ?>
        <form method="post" class="card">
            <?php csrf_field(); ?>
            <h2>Вход</h2>
            <label>Email<input type="email" name="email" required></label>
            <label>Пароль<input type="password" name="password" required></label>
            <button class="btn primary" type="submit">Войти</button>
            <p class="muted">Нет аккаунта? <a href="?action=register">Регистрация</a></p>
        </form>
        <?php
    });
}

function handle_logout(): void {
    session_destroy();
    redirect('?action=login');
}

// ---------------------- PROJECTS ---------------------- //
function handle_projects_list(): void {
    $u = require_login();
    $db = db();
    $stmt = $db->query('SELECT p.*, u.name AS owner_name FROM projects p JOIN users u ON u.id = p.owner_id ORDER BY p.created_at DESC');
    $projects = $stmt->fetchAll();

    render('Проекты', function () use ($projects) {
        ?>
        <div class="toolbar">
            <a class="btn primary" href="?action=project_new">+ Новый проект</a>
        </div>
        <div class="grid">
            <?php foreach ($projects as $p): ?>
                <a class="card link-card" href="?action=project_view&id=<?= (int)$p['id'] ?>">
                    <h3><?= e($p['name']) ?></h3>
                    <p class="muted">Автор: <?= e($p['owner_name']) ?> • <?= e($p['created_at']) ?></p>
                    <p><?= nl2br(e($p['description'])) ?></p>
                </a>
            <?php endforeach; ?>
        </div>
        <?php
    });
}

function handle_project_new(): void {
    require_login();
    if (is_post()) {
        check_csrf();
        $name = trim((string)($_POST['name'] ?? ''));
        $desc = trim((string)($_POST['description'] ?? ''));
        if (!$name) { flash('Название обязательно', 'danger'); }
        else {
            $stmt = db()->prepare('INSERT INTO projects(name,description,owner_id,created_at) VALUES(?,?,?,?)');
            $stmt->execute([$name, $desc, current_user()['id'], now()]);
            $pid = (int)db()->lastInsertId();
            log_event(db(), 'project', $pid, 'create', ['name' => $name]);
            redirect('?action=project_view&id=' . $pid);
        }
    }
    render('Новый проект', function () {
        ?>
        <form method="post" class="card">
            <?php csrf_field(); ?>
            <h2>Новый проект</h2>
            <label>Название<input type="text" name="name" required></label>
            <label>Описание<textarea name="description" rows="4"></textarea></label>
            <button class="btn primary">Создать</button>
        </form>
        <?php
    });
}

function find_project(int $id): array {
    $stmt = db()->prepare('SELECT p.*, u.name AS owner_name FROM projects p JOIN users u ON u.id=p.owner_id WHERE p.id=?');
    $stmt->execute([$id]);
    $p = $stmt->fetch();
    if (!$p) { http_response_code(404); echo 'Проект не найден'; exit; }
    return $p;
}

function handle_project_view(): void {
    $u = require_login();
    $id = (int)($_GET['id'] ?? 0);
    $project = find_project($id);

    // Фильтры задач
    $status = $_GET['status'] ?? '';
    $assignee = (int)($_GET['assignee'] ?? 0);
    $q = trim((string)($_GET['q'] ?? ''));

    $db = db();
    $sql = 'SELECT t.*, r.name AS reporter_name, a.name AS assignee_name FROM tasks t 
            JOIN users r ON r.id = t.reporter_id 
            LEFT JOIN users a ON a.id = t.assignee_id 
            WHERE t.project_id = ?';
    $params = [$id];
    if ($status !== '') { $sql .= ' AND t.status = ?'; $params[] = $status; }
    if ($assignee > 0) { $sql .= ' AND t.assignee_id = ?'; $params[] = $assignee; }
    if ($q !== '') { $sql .= ' AND (t.title LIKE ? OR t.description LIKE ? OR t.labels LIKE ?)'; $like = '%' . $q . '%'; $params[] = $like; $params[] = $like; $params[] = $like; }
    $sql .= ' ORDER BY CASE t.status WHEN "todo" THEN 0 WHEN "in_progress" THEN 1 WHEN "blocked" THEN 2 ELSE 3 END, 
                        CASE t.priority WHEN "urgent" THEN 0 WHEN "high" THEN 1 WHEN "normal" THEN 2 ELSE 3 END, t.due_date IS NULL, t.due_date';
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $tasks = $stmt->fetchAll();

    $users = $db->query('SELECT id,name FROM users ORDER BY name')->fetchAll();

    render('Проект: ' . $project['name'], function () use ($project, $tasks, $users, $status, $assignee, $q) {
        ?>
        <div class="toolbar">
            <a class="btn" href="?action=projects">← Все проекты</a>
            <a class="btn primary" href="?action=task_new&project_id=<?= (int)$project['id'] ?>">+ Новая задача</a>
        </div>
        <div class="card">
            <h2><?= e($project['name']) ?></h2>
            <p class="muted">Владелец: <?= e($project['owner_name']) ?> • <?= e($project['created_at']) ?></p>
            <p><?= nl2br(e($project['description'])) ?></p>
        </div>

        <form class="filters" method="get">
            <input type="hidden" name="action" value="project_view">
            <input type="hidden" name="id" value="<?= (int)$project['id'] ?>">
            <input type="search" name="q" placeholder="Поиск" value="<?= e($q) ?>">
            <select name="status">
                <option value="">Статус: любой</option>
                <?php foreach (["todo"=>"К выполнению","in_progress"=>"В работе","blocked"=>"Заблокирована","done"=>"Готово"] as $k=>$v): ?>
                    <option value="<?= e($k) ?>" <?= $status===$k?'selected':'' ?>><?= e($v) ?></option>
                <?php endforeach; ?>
            </select>
            <select name="assignee">
                <option value="0">Исполнитель: любой</option>
                <?php foreach ($users as $u): ?>
                    <option value="<?= (int)$u['id'] ?>" <?= $assignee===(int)$u['id']?'selected':'' ?>><?= e($u['name']) ?></option>
                <?php endforeach; ?>
            </select>
            <button class="btn">Фильтр</button>
        </form>

        <div class="kanban">
            <?php
            $columns = [
                'todo' => 'К выполнению',
                'in_progress' => 'В работе',
                'blocked' => 'Блок',
                'done' => 'Готово'
            ];
            $grouped = [];
            foreach ($columns as $key=>$_) $grouped[$key] = [];
            foreach ($tasks as $t) { $grouped[$t['status']][] = $t; }
            ?>
            <?php foreach ($columns as $key=>$label): ?>
                <div class="column">
                    <h3><?= e($label) ?></h3>
                    <?php foreach ($grouped[$key] as $t): ?>
                        <a class="card task-card link-card" href="?action=task_view&id=<?= (int)$t['id'] ?>">
                            <div class="task-title"><?= e($t['title']) ?></div>
                            <div class="task-meta">
                                <span class="chip pr-<?= e($t['priority']) ?>"><?= e($t['priority']) ?></span>
                                <?php if ($t['due_date']): ?><span class="chip">Срок: <?= e($t['due_date']) ?></span><?php endif; ?>
                                <?php if ($t['assignee_name']): ?><span class="chip">→ <?= e($t['assignee_name']) ?></span><?php endif; ?>
                            </div>
                            <?php if ($t['labels']): ?><div class="muted small">#<?= e($t['labels']) ?></div><?php endif; ?>
                        </a>
                    <?php endforeach; ?>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
    });
}

// ---------------------- TASKS ---------------------- //
function users_list(): array {
    return db()->query('SELECT id, name FROM users ORDER BY name')->fetchAll();
}

function handle_task_new(): void {
    $u = require_login();
    $project_id = (int)($_GET['project_id'] ?? 0);
    if (!$project_id) redirect('?action=projects');
    if (is_post()) {
        check_csrf();
        $title = trim((string)($_POST['title'] ?? ''));
        $desc = trim((string)($_POST['description'] ?? ''));
        $status = $_POST['status'] ?? 'todo';
        $priority = $_POST['priority'] ?? 'normal';
        $due_date = $_POST['due_date'] ?: null;
        $assignee_id = (int)($_POST['assignee_id'] ?? 0) ?: null;
        $labels = trim((string)($_POST['labels'] ?? ''));
        if (!$title) { flash('Название обязательно', 'danger'); }
        else {
            $stmt = db()->prepare('INSERT INTO tasks(project_id,title,description,status,priority,due_date,reporter_id,assignee_id,labels,created_at) VALUES(?,?,?,?,?,?,?,?,?,?)');
            $stmt->execute([$project_id,$title,$desc,$status,$priority,$due_date,current_user()['id'],$assignee_id,$labels,now()]);
            $tid = (int)db()->lastInsertId();
            log_event(db(), 'task', $tid, 'create', ['title'=>$title]);
            redirect('?action=task_view&id=' . $tid);
        }
    }
    $users = users_list();
    render('Новая задача', function () use ($project_id, $users) {
        ?>
        <form method="post" class="card">
            <?php csrf_field(); ?>
            <h2>Новая задача</h2>
            <input type="hidden" name="project_id" value="<?= (int)$project_id ?>">
            <label>Заголовок<input type="text" name="title" required></label>
            <label>Описание<textarea name="description" rows="5"></textarea></label>
            <div class="row">
                <label>Статус
                    <select name="status">
                        <?php foreach (["todo"=>"К выполнению","in_progress"=>"В работе","blocked"=>"Заблокирована","done"=>"Готово"] as $k=>$v): ?>
                            <option value="<?= e($k) ?>"><?= e($v) ?></option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label>Приоритет
                    <select name="priority">
                        <?php foreach (["low"=>"Низкий","normal"=>"Нормальный","high"=>"Высокий","urgent"=>"Срочный"] as $k=>$v): ?>
                            <option value="<?= e($k) ?>"><?= e($v) ?></option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label>Срок<input type="date" name="due_date"></label>
            </div>
            <label>Исполнитель
                <select name="assignee_id">
                    <option value="">— не назначен —</option>
                    <?php foreach ($users as $usr): ?>
                        <option value="<?= (int)$usr['id'] ?>"><?= e($usr['name']) ?></option>
                    <?php endforeach; ?>
                </select>
            </label>
            <label>Метки (через запятую)<input type="text" name="labels" placeholder="bug, backend, ui"></label>
            <button class="btn primary">Создать</button>
        </form>
        <?php
    });
}

function find_task(int $id): array {
    $stmt = db()->prepare('SELECT t.*, p.name AS project_name FROM tasks t JOIN projects p ON p.id=t.project_id WHERE t.id=?');
    $stmt->execute([$id]);
    $t = $stmt->fetch();
    if (!$t) { http_response_code(404); echo 'Задача не найдена'; exit; }
    return $t;
}

function handle_task_view(): void {
    $u = require_login();
    $id = (int)($_GET['id'] ?? 0);
    $t = find_task($id);

    // Добавление комментария
    if (is_post() && isset($_POST['add_comment'])) {
        check_csrf();
        $body = trim((string)($_POST['body'] ?? ''));
        if ($body) {
            $stmt = db()->prepare('INSERT INTO comments(task_id,author_id,body,created_at) VALUES(?,?,?,?)');
            $stmt->execute([$t['id'], current_user()['id'], $body, now()]);
            log_event(db(), 'task', $t['id'], 'comment', []);
            flash('Комментарий добавлен', 'success');
            redirect('?action=task_view&id=' . $t['id']);
        } else { flash('Пустой комментарий', 'danger'); }
    }

    // Обновление основной информации задачи
    if (is_post() && isset($_POST['update_task'])) {
        check_csrf();
        $title = trim((string)($_POST['title'] ?? ''));
        $description = trim((string)($_POST['description'] ?? ''));
        $status = (string)($_POST['status'] ?? 'todo');
        $priority = (string)($_POST['priority'] ?? 'normal');
        $due_date = $_POST['due_date'] ?: null;
        $assignee_id = (int)($_POST['assignee_id'] ?? 0) ?: null;
        $labels = trim((string)($_POST['labels'] ?? ''));

        $stmt = db()->prepare('UPDATE tasks SET title=?, description=?, status=?, priority=?, due_date=?, assignee_id=?, labels=?, updated_at=? WHERE id=?');
        $stmt->execute([$title,$description,$status,$priority,$due_date,$assignee_id,$labels,now(),$t['id']]);
        log_event(db(), 'task', $t['id'], 'update', ['status'=>$status,'priority'=>$priority]);
        flash('Задача обновлена', 'success');
        redirect('?action=task_view&id=' . $t['id']);
    }

    $users = users_list();
    $comments = db()->prepare('SELECT c.*, u.name AS author_name FROM comments c JOIN users u ON u.id=c.author_id WHERE task_id=? ORDER BY c.created_at');
    $comments->execute([$t['id']]);
    $comments = $comments->fetchAll();

    render('Задача #' . $t['id'], function () use ($t, $users, $comments) {
        ?>
        <div class="toolbar">
            <a class="btn" href="?action=project_view&id=<?= (int)$t['project_id'] ?>">← К проекту: <?= e($t['project_name']) ?></a>
        </div>
        <form class="card" method="post">
            <?php csrf_field(); ?>
            <h2>#<?= (int)$t['id'] ?> — <input class="inline" type="text" name="title" value="<?= e($t['title']) ?>" required></h2>
            <label>Описание<textarea name="description" rows="6"><?= e($t['description']) ?></textarea></label>
            <div class="row">
                <label>Статус
                    <select name="status">
                        <?php foreach (["todo"=>"К выполнению","in_progress"=>"В работе","blocked"=>"Заблокирована","done"=>"Готово"] as $k=>$v): ?>
                            <option value="<?= e($k) ?>" <?= option_selected($k, $t['status']) ?>><?= e($v) ?></option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label>Приоритет
                    <select name="priority">
                        <?php foreach (["low"=>"Низкий","normal"=>"Нормальный","high"=>"Высокий","urgent"=>"Срочный"] as $k=>$v): ?>
                            <option value="<?= e($k) ?>" <?= option_selected($k, $t['priority']) ?>><?= e($v) ?></option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label>Срок<input type="date" name="due_date" value="<?= e((string)$t['due_date']) ?>"></label>
                <label>Исполнитель
                    <select name="assignee_id">
                        <option value="">— не назначен —</option>
                        <?php foreach ($users as $usr): ?>
                            <option value="<?= (int)$usr['id'] ?>" <?= ((int)$t['assignee_id'] === (int)$usr['id'])? 'selected':'' ?>><?= e($usr['name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </label>
            </div>
            <label>Метки<input type="text" name="labels" value="<?= e((string)$t['labels']) ?>"></label>
            <button class="btn primary" name="update_task" value="1">Сохранить изменения</button>
        </form>

        <div class="card">
            <h3>Комментарии</h3>
            <?php foreach ($comments as $c): ?>
                <div class="comment">
                    <div class="comment-meta">
                        <strong><?= e($c['author_name']) ?></strong>
                        <span class="muted">• <?= e($c['created_at']) ?></span>
                    </div>
                    <div><?= nl2br(e($c['body'])) ?></div>
                </div>
            <?php endforeach; ?>
            <form method="post" class="mt">
                <?php csrf_field(); ?>
                <textarea name="body" rows="3" placeholder="Написать комментарий…"></textarea>
                <button class="btn" name="add_comment" value="1">Отправить</button>
            </form>
        </div>
        <?php
    });
}

// ---------------------- HOME/DASHBOARD ---------------------- //
function handle_home(): void {
    $u = current_user();
    if (!$u) { redirect('?action=login'); }

    // Небольшая сводка: мои задачи
    $db = db();
    $my = $db->prepare('SELECT t.*, p.name AS project_name FROM tasks t JOIN projects p ON p.id=t.project_id WHERE assignee_id=? ORDER BY t.due_date IS NULL, t.due_date');
    $my->execute([$u['id']]);
    $myTasks = $my->fetchAll();

    // Недавние события
    $events = $db->query('SELECT e.*, u.name AS actor_name FROM events e LEFT JOIN users u ON u.id=e.actor_id ORDER BY e.created_at DESC LIMIT 20')->fetchAll();

    render('Обзор', function () use ($myTasks, $events) {
        ?>
        <div class="grid">
            <div class="card">
                <h2>Мои задачи</h2>
                <?php if (!$myTasks): ?>
                    <p class="muted">Пока пусто</p>
                <?php else: ?>
                    <ul class="list">
                        <?php foreach ($myTasks as $t): ?>
                            <li>
                                <a href="?action=task_view&id=<?= (int)$t['id'] ?>">#<?= (int)$t['id'] ?> — <?= e($t['title']) ?></a>
                                <span class="muted">(<?= e($t['project_name']) ?>)</span>
                                <?php if ($t['due_date']): ?><span class="chip">до <?= e($t['due_date']) ?></span><?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            <div class="card">
                <h2>Лента событий</h2>
                <ul class="list">
                    <?php foreach ($events as $ev): ?>
                        <li class="event">
                            <span class="muted">[<?= e($ev['created_at']) ?>]</span>
                            <strong><?= e($ev['actor_name'] ?? 'система') ?></strong>
                            <span><?= e($ev['action']) ?></span>
                            <span class="muted">→ <?= e($ev['entity']) ?> #<?= (int)$ev['entity_id'] ?></span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        </div>
        <?php
    });
}

// ---------------------- RENDERING ---------------------- //
function csrf_field(): void { echo '<input type="hidden" name="csrf" value="' . e(csrf_token()) . '">'; }

function render(string $title, callable $content): void {
    $user = current_user();
    $flashes = flash();
    ?><!doctype html>
<html lang="ru">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title><?= e(APP_NAME . ' — ' . $title) ?></title>
    <style>
        :root { --bg:#0b0d10; --fg:#e7e9ee; --muted:#9aa4b2; --card:#12151a; --border:#222833; --primary:#4f8cff; --danger:#ff5d5d; --ok:#33d69f; }
        *{box-sizing:border-box}
        body{margin:0; background:var(--bg); color:var(--fg); font:16px/1.5 system-ui, -apple-system, Segoe UI, Roboto, Ubuntu, Cantarell, Noto Sans, Arial, sans-serif}
        a{color:var(--fg); text-decoration:none}
        header{display:flex; align-items:center; justify-content:space-between; padding:16px 20px; border-bottom:1px solid var(--border); position:sticky; top:0; backdrop-filter:saturate(180%) blur(8px); background:rgba(11,13,16,.75)}
        .brand{display:flex; gap:10px; align-items:center}
        .brand .logo{width:28px; height:28px; border-radius:8px; background:linear-gradient(135deg,var(--primary),#9b7dff); display:inline-block}
        .container{max-width:1100px; margin:0 auto; padding:20px}
        .btn{display:inline-flex; align-items:center; gap:8px; background:transparent; border:1px solid var(--border); color:var(--fg); padding:10px 14px; border-radius:12px; cursor:pointer}
        .btn:hover{border-color:#3a4250}
        .btn.primary{background:var(--primary); border-color:transparent; color:white}
        .btn.primary:hover{filter:brightness(1.05)}
        .btn.danger{background:var(--danger); border-color:transparent}
        .toolbar{display:flex; gap:10px; margin-bottom:16px}
        .card{background:var(--card); border:1px solid var(--border); border-radius:16px; padding:16px; margin:12px 0}
        .grid{display:grid; grid-template-columns:repeat(auto-fill,minmax(260px,1fr)); gap:14px}
        .link-card{transition:transform .06s ease;}
        .link-card:hover{transform:translateY(-2px)}
        label{display:flex; flex-direction:column; gap:6px; margin:8px 0; font-size:14px}
        input, select, textarea{border:1px solid var(--border); background:#0e1116; color:var(--fg); border-radius:12px; padding:10px 12px; outline:none}
        input.inline{display:inline-block; width:70%}
        textarea{resize:vertical}
        .muted{color:var(--muted)} .small{font-size:12px}
        .row{display:grid; grid-template-columns:repeat(4,1fr); gap:10px}
        .filters{display:flex; flex-wrap:wrap; gap:10px; align-items:center; background:var(--card); border:1px solid var(--border); border-radius:16px; padding:12px; margin:12px 0}
        .kanban{display:grid; grid-template-columns:repeat(4,1fr); gap:12px; margin-top:12px}
        .column{background:var(--card); border:1px solid var(--border); border-radius:16px; padding:12px; min-height:80px}
        .task-card .task-title{font-weight:600; margin-bottom:6px}
        .task-meta{display:flex; gap:6px; flex-wrap:wrap; margin:6px 0}
        .chip{display:inline-block; padding:2px 8px; border-radius:999px; border:1px solid var(--border)}
        .pr-low{border-color:#2e5}
        .pr-normal{border-color:#59f}
        .pr-high{border-color:#fc5}
        .pr-urgent{border-color:#f66}
        .list{list-style:none; padding:0; margin:0}
        .list li{padding:8px 0; border-bottom:1px solid var(--border)}
        .comment{padding:10px 0; border-top:1px solid var(--border)}
        .mt{margin-top:10px}
        .flash{padding:10px 14px; border-radius:12px; margin:8px 0; border:1px solid var(--border)}
        .flash.success{background:#103822; border-color:#1a4}
        .flash.danger{background:#3a1212; border-color:#a33}
        .flash.info{background:#152235; border-color:#345}
        @media (max-width:900px){ .row{grid-template-columns:1fr 1fr}; .kanban{grid-template-columns:1fr 1fr} }
        @media (max-width:600px){ .row{grid-template-columns:1fr}; .kanban{grid-template-columns:1fr} }
    </style>
</head>
<body>
<header>
    <div class="brand">
        <span class="logo"></span>
        <a href="index.php"><strong><?= e(APP_NAME) ?></strong></a>
    </div>
    <nav>
        <?php if ($user): ?>
            <a class="btn" href="?action=projects">Проекты</a>
            <span class="muted">|</span>
            <span class="muted"><?= e($user['name']) ?></span>
            <a class="btn" href="?action=logout">Выйти</a>
        <?php else: ?>
            <a class="btn" href="?action=login">Войти</a>
        <?php endif; ?>
    </nav>
</header>
<div class="container">
    <?php if ($flashes): foreach ($flashes as [$type,$msg]): ?>
        <div class="flash <?= e($type) ?>"><?= e($msg) ?></div>
    <?php endforeach; endif; ?>
    <?php $content(); ?>
</div>
</body>
</html>
<?php }

// ---------------------- ROUTER ---------------------- //
$action = $_GET['action'] ?? 'home';

switch ($action) {
    case 'register': handle_register(); break;
    case 'login': handle_login(); break;
    case 'logout': handle_logout(); break;
    case 'projects': handle_projects_list(); break;
    case 'project_new': handle_project_new(); break;
    case 'project_view': handle_project_view(); break;
    case 'task_new': handle_task_new(); break;
    case 'task_view': handle_task_view(); break;
    case 'home': default: handle_home(); break;
}

?>